package com.roid.bettingtips

import android.content.Intent
import androidx.appcompat.app.AppCompatActivity
import android.os.Bundle
import android.text.TextUtils
import android.text.method.LinkMovementMethod
import android.util.Log
import android.widget.Button
import android.widget.EditText
import android.widget.TextView
import androidx.cardview.widget.CardView
import com.facebook.AccessToken
import com.facebook.CallbackManager
import com.facebook.FacebookCallback
import com.facebook.FacebookException
import com.facebook.login.LoginManager
import com.facebook.login.LoginResult
import com.facebook.login.widget.LoginButton
import com.firebase.ui.auth.IdpResponse
import com.google.android.gms.auth.api.signin.GoogleSignIn
import com.google.android.gms.auth.api.signin.GoogleSignInClient
import com.google.android.gms.auth.api.signin.GoogleSignInOptions
import com.google.android.gms.common.api.ApiException
import com.google.android.gms.tasks.OnCompleteListener
import com.google.firebase.auth.FacebookAuthProvider
import com.google.firebase.auth.FirebaseAuth
import com.google.firebase.auth.FirebaseAuthUserCollisionException
import com.google.firebase.auth.GoogleAuthProvider
import com.google.firebase.auth.ktx.auth
import com.google.firebase.database.DatabaseReference
import com.google.firebase.database.ktx.database
import com.google.firebase.ktx.Firebase
import com.tapadoo.alerter.Alerter
import com.roid.bettingtips.data.UserData
import okhttp3.*
import org.json.JSONException
import org.json.JSONObject
import java.io.IOException

class SignActivity : AppCompatActivity() {

    private val client = OkHttpClient()
    private var serverUrl: String = "https://api.roidbetting.online/getId"
    private lateinit var googleSignInClient: GoogleSignInClient
    private lateinit var loginWgoogle: CardView
    private lateinit var loginWfacebook: CardView
    private lateinit var loginWmail: CardView
    private lateinit var fbBtn: LoginButton
    private lateinit var callbackManager: CallbackManager
    private lateinit var database: DatabaseReference
    private lateinit var auth: FirebaseAuth
    private var loadingDialog: LoadingDialog = LoadingDialog()
    private lateinit var emailInp: EditText
    private lateinit var passInp: EditText
    private lateinit var signinBtn: Button

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_sign)

        auth = Firebase.auth
        database = Firebase.database.reference

        val gso = GoogleSignInOptions.Builder(GoogleSignInOptions.DEFAULT_SIGN_IN)
            .requestIdToken(getString(R.string.default_web_client_id))
            .requestEmail()
            .build()

        callbackManager = CallbackManager.Factory.create()
        LoginManager.getInstance().logOut()

        googleSignInClient = GoogleSignIn.getClient(this, gso)

        loginWgoogle = findViewById(R.id.loginWGoogle)
        loginWfacebook = findViewById(R.id.loginWfacebook)
        fbBtn = findViewById(R.id.fb_btn)
        loginWmail = findViewById(R.id.loginWMail)
        emailInp = findViewById(R.id.email)
        passInp = findViewById(R.id.password)
        signinBtn = findViewById(R.id.signinBtn)

        signinBtn.setOnClickListener(){
            val email = emailInp.text.toString()
            val password = passInp.text.toString()

            if(email.isEmailValid()){
                if(password != ""){
                    showProgressBar()
                    auth.signInWithEmailAndPassword(email, password).addOnCompleteListener(this, OnCompleteListener { task ->
                        hideProgressBar()
                        if (task.isSuccessful) {
                            val intent = Intent(baseContext, MainActivity::class.java)
                            startActivity(intent)
                            finish()
                        } else {
                            errorAlert()
                            passInp.text.clear()
                        }
                    })
                }else{
                    showInfoMsg(getString(R.string.error))
                }
            }else{
                showInfoMsg(getString(R.string.emptyError))
            }
        }

        loginWgoogle.setOnClickListener(){
            signIn()
        }

        loginWfacebook.setOnClickListener(){
            fbBtn.performClick()
        }

        loginWmail.setOnClickListener(){
            val intent = Intent(baseContext, EmailSignActivity::class.java)
            startActivity(intent)
            finish()
        }

        fbBtn.setReadPermissions("email", "public_profile")

        fbBtn.registerCallback(callbackManager, object : FacebookCallback<LoginResult> {
            override fun onSuccess(loginResult: LoginResult) {
                Log.d(TAG, "facebook:onSuccess:$loginResult")
                handleFacebookAccessToken(loginResult.accessToken)
            }

            override fun onCancel() {
            }

            override fun onError(error: FacebookException) {
                errorAlert()
            }
        })

        val regTextBtm = findViewById<TextView>(R.id.regTextBottom)
        regTextBtm.movementMethod = LinkMovementMethod.getInstance()

    }

    private fun signIn() {
        val signInIntent = googleSignInClient.signInIntent
        startActivityForResult(signInIntent, RC_SIGN_IN)
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)

        if (requestCode == RC_SIGN_IN) {
            val task = GoogleSignIn.getSignedInAccountFromIntent(data)
            val response = IdpResponse.fromResultIntent(data)
            try {
                val account = task.getResult(ApiException::class.java)!!
                Log.d(TAG, "firebaseAuthWithGoogle:" + account.id)
                firebaseAuthWithGoogle(account.idToken!!)
            } catch (e: ApiException) {
                errorAlert()
            }
        }

        callbackManager.onActivityResult(requestCode, resultCode, data)
    }

    private fun handleFacebookAccessToken(token: AccessToken) {
        Log.d(TAG, "handleFacebookAccessToken:$token")

        val credential = FacebookAuthProvider.getCredential(token.token)
        auth.signInWithCredential(credential)
            .addOnCompleteListener(this) { task ->
                if (task.isSuccessful) {
                    Log.d(TAG, "signInWithCredential:success")
                    val user = auth.currentUser

                    if(task.result?.additionalUserInfo?.isNewUser == true && user != null){
                        newUserSave(user.uid, user.displayName, user.email)
                    }else{
                        hideProgressBar()
                        val intent = Intent(baseContext, MainActivity::class.java)
                        startActivity(intent)
                        finish()
                    }
                } else {
                    if(task.exception is FirebaseAuthUserCollisionException){
                        showInfoMsg(getString(R.string.emailDuplicateError))
                    }else{
                        errorAlert()
                    }
                    LoginManager.getInstance().logOut()
                }
            }
    }

    private fun firebaseAuthWithGoogle(idToken: String) {
        showProgressBar()
        val credential = GoogleAuthProvider.getCredential(idToken, null)
        auth.signInWithCredential(credential)
            .addOnCompleteListener(this) { task ->
                if (task.isSuccessful) {
                    Log.d(TAG, "signInWithCredential:success")
                    val user = auth.currentUser
                    if(task.result?.additionalUserInfo?.isNewUser == true && user != null){
                        newUserSave(user.uid, user.displayName, user.email)
                    }else{
                        hideProgressBar()
                        val intent = Intent(baseContext, MainActivity::class.java)
                        startActivity(intent)
                        finish()
                    }

                } else {
                    errorAlert()
                }
                hideProgressBar()
            }
    }

    private fun newUserSave(userId: String, displayName: String, email: String?) {
        val url = serverUrl

        val request = Request.Builder()
            .url(url)
            .build()

        client.newCall(request).enqueue(object : Callback {
            override fun onFailure(call: Call, e: IOException) {
                runOnUiThread {
                    errorAlert()
                }
            }

            override fun onResponse(call: Call, response: Response) {
                runOnUiThread {
                    try {
                        val strResponse = response.body()!!.string()
                        val json = JSONObject(strResponse)

                        if (json.has("id")) {
                            val id = json.getInt("id")
                            val name = displayName.split(" ").toTypedArray()
                            var firstName = ""
                            var lastName = ""

                            if(name.getOrNull(0) != null){firstName = name[0]}
                            if(name.getOrNull(1) != null){lastName = name[1]}

                            val prize = if(json.has("register_prize")){
                                json.getInt("register_prize")
                            }else{
                                0
                            }

                            val user = UserData(id, firstName, lastName, email, prize,0,0,0,0, "0","0","0","0", "0")

                            database.child("users").child(userId).setValue(user)
                                .addOnSuccessListener {
                                    hideProgressBar()
                                    val intent = Intent(baseContext, MainActivity::class.java)
                                    startActivity(intent)
                                    finish()
                                }
                                .addOnFailureListener {
                                    errorAlert()
                                }
                        }else{
                            errorAlert()
                        }
                    } catch (jpe: JSONException) {
                        errorAlert()
                        Log.d(TAG, jpe.toString())
                    } catch (ioe: IOException) {
                        errorAlert()
                        Log.d(TAG, ioe.toString())
                    }
                }
            }
        })
    }

    private fun showInfoMsg(infoMsg: String){
        Alerter.create(this)
            .setTitle(getString(R.string.info))
            .setText(infoMsg)
            .setIcon(R.drawable.ic_error)
            .setBackgroundColorRes(R.color.info_color)
            .show()
    }

    private fun errorAlert(){
        Alerter.create(this)
            .setTitle(getString(R.string.error))
            .setText(getString(R.string.loginFail))
            .setIcon(R.drawable.ic_error)
            .setBackgroundColorRes(R.color.error_color)
            .show()
    }


    private fun showProgressBar(){
        if(!loadingDialog.isVisible){
            loadingDialog.show(supportFragmentManager, "loadingDialog")
        }
    }

    private fun hideProgressBar(){
        if(loadingDialog.isVisible){
            loadingDialog.dismiss()
        }
    }

    fun String.isEmailValid(): Boolean {
        return !TextUtils.isEmpty(this) && android.util.Patterns.EMAIL_ADDRESS.matcher(this).matches()
    }

    companion object {
        private const val TAG = "GoogleActivity"
        private const val RC_SIGN_IN = 9001
    }
}